//	TorusGames2DGomoku.c
//
//	The board's n² cells (n = GOMOKU_SIZE) are indexed as
//
//		( 0, n-1)	( 1, n-1)	( 2, n-1)	…	(n-1,n-1)
//			⋮			⋮			⋮		⋰		⋮
//		( 0,  2 )	( 1,  2 )	( 2,  2 )	…	(n-1, 2 )
//		( 0,  1 )	( 1,  1 )	( 2,  1 )	…	(n-1, 1 )
//		( 0,  0 )	( 1,  0 )	( 2,  0 )	…	(n-1, 0 )
//
//	to agree with Metal's bottom-up Normalized Device Coordinates,
//	at the expense of violating the usual (row, column)
//	convention for arrays.  Cell (0,0) straddles all four corners.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesLocalization.h"
#include "GeometryGamesSound.h"
#include <math.h>


//	Let the background repeat in a 5×5 pattern,
//	to avoid obvious repetitions on the 6×6 board.
#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_GOMOKU	5

//	A halo marks the winning five-in-a-row.
//	Let it extend beyond the width of the stones themselves.
//	In fact, the current texture is highly transparent
//	and needs to extend well beyond the stones.
#define HALO_FACTOR	1.5

//	Note:  A 6×6 board contains 6 * 6 * 4 = 144 five-in-a-row lines.
//	So if each value in the sequence below is at least 144 times
//	as great as the number below it, that will ensure
//	(for better or worse) that a higher-occupancy row is always
//	more important than any combination of lower-occupancy rows.
#define GOMOKU_WIN_BASE_VALUE	((signed int)0x01000000)
#define GOMOKU_4_OF_5_VALUE		((signed int)0x00010000)
#define GOMOKU_3_OF_5_VALUE		((signed int)0x00000100)
#define GOMOKU_2_OF_5_VALUE		((signed int)0x00000001)

//	Caution:  Don't set
//
//		GOMOKU_INITIAL_ALPHA = smallest signed int
//							 = -2147483648
//
//	because when we do the swap
//
//		α' = -β
//		β' = -α
//
//	we'd get
//
//		β' = -(-2147483648)
//		   = +2147483648  (overflows range for positive signed ints)
//		   = -2147483648  (re-interpreted modulo 2³²)
//
//	In other words, as a signed int, -2147483648 is its own negative!
//
#define GOMOKU_LARGE_NEGATIVE_VALUE	(-(signed int)0x10000000)
#define GOMOKU_INITIAL_ALPHA		(-(signed int)0x10000000)	//	see note above
#define GOMOKU_INITIAL_BETA			( (signed int)0x10000000)


//	Each potential winning line is a 5-element set of intersections,
//	which we'll describe by its center and direction.
//	Antipodal directions are equivalent,
//	so we need consider only four directions, not eight.
//
//	In the case of a Klein bottle, the sense of "east" is relative
//	to the fundamental square, and is not consistent across the Klein bottle itself,
//	but that's OK.  For a given winning line, the centerpoint
//	takes its sense of "east" from the fundamental square,
//	and extends that sense of "east" consistently to the rest
//	of the line (even though at non-centerpoints the winning line's
//	sense of east may disagree with that of the fundamental square).
static const unsigned int	gGomokuDirections[4][2] =
							{
								{+1,  0},	//	east
								{+1, +1},	//	northeast
								{ 0, +1},	//	north
								{-1, +1}	//	northwest
							};


//	Public functions with private names
static void	GomokuShutDown(ModelData *md);
static void	GomokuReset(ModelData *md);
static bool	GomokuDragBegin(ModelData *md, bool aRightClick);
static void	GomokuDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void	GomokuDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
static void	GomokuSimulationUpdate(ModelData *md);

//	Private functions
static void			InitEmptyBoard(GomokuBoard *aBoard, TopologyType aTopology);
static void			AdvanceOneStep(signed int *hh, signed int *vv, signed int *dh, signed int dv, TopologyType aTopology);
static void			AddOrRemoveStone(GomokuBoard *aBoard, unsigned int h, unsigned int v, GomokuPlayer aNewStone);
static void			FindHitIntersection(Placement2D *aHandPlacement, TopologyType aTopology, unsigned int *h, unsigned int *v);
static GomokuPlayer	CheckForWin(ModelData *md, Placement2D *aFiveInARow);
static bool			LocateWinLine(GomokuBoard *aBoard, unsigned int aNumBlackStones, unsigned int aNumWhiteStones, Placement2D *aFiveInARow);
static bool			BoardIsFull(ModelData *md);
static bool			BoardIsEmpty(ModelData *md);
static void			ComputerMoves(ModelData *md);
static void			ComputerSelectsMove(ModelData *md);
static unsigned int	ValueOfSituation(GomokuBoard *aBoard, GomokuPlayer aPlayer, unsigned int aRecursionHeight,
						signed int anAlpha, signed int aBeta, bool *anAbortFlag,
						unsigned int *aBestMoveH, unsigned int *aBestMoveV);
static signed int	ValueOfBoardAtLeafNode(GomokuBoard *aBoard, GomokuPlayer aPlayer);
static void			ComputerMakesMove(ModelData *md);
static void			TerminateThinkingThread(ModelData *md);


void Gomoku2DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= &GomokuShutDown;
	md->itsGameReset					= &GomokuReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= &GomokuDragBegin;
	md->itsGame2DDragObject				= &GomokuDragObject;
	md->itsGame2DDragEnd				= &GomokuDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= &GomokuSimulationUpdate;
	md->itsGameRefreshMessage			= NULL;

	//	Initialize variables.
	//
	//	GomokuReset() will initialize itsBoard, itsWhoseTurn and itsGameStatus.

	md->itsGameOf.Gomoku2D.itsHitIntersectionH = 0;
	md->itsGameOf.Gomoku2D.itsHitIntersectionV = 0;

	md->itsGameOf.Gomoku2D.itsWinLine.itsH		= 0.0;
	md->itsGameOf.Gomoku2D.itsWinLine.itsV		= 0.0;
	md->itsGameOf.Gomoku2D.itsWinLine.itsFlip	= false;
	md->itsGameOf.Gomoku2D.itsWinLine.itsAngle	= 0.0;
	md->itsGameOf.Gomoku2D.itsWinLine.itsSizeH	= 0.0;
	md->itsGameOf.Gomoku2D.itsWinLine.itsSizeV	= 0.0;
	
	//	No thinking thread yet.
	md->itsGameOf.Gomoku2D.itsThreadThinkingFlag	= false;
	md->itsGameOf.Gomoku2D.itsThreadAbortFlag		= false;

	//	Set up the board.
	GomokuReset(md);
}


static void GomokuShutDown(ModelData *md)
{
	//	Terminate the thinking thread, if one is active.
	TerminateThinkingThread(md);
	
	//	Cancel any pending simulation.
	SimulationEnd(md);
}


static void GomokuReset(ModelData *md)
{
	//	Abort any pending computer move
	//	and/or any pending simulation.
	TerminateThinkingThread(md);
	SimulationEnd(md);

	InitEmptyBoard(&md->itsGameOf.Gomoku2D.itsBoard, md->itsTopology);

	md->itsGameIsOver						= false;
	md->itsFlashFlag						= false;
	md->itsGameOf.Gomoku2D.itsWhoseTurn		= GomokuPlayerBlack;
	md->itsGameOf.Gomoku2D.itsGameStatus	= GomokuGameInProgress;

#ifdef GAME_CONTENT_FOR_SCREENSHOT

	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 0, 4, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 1, 5, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 2, 0, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 2, 2, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 4, 5, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 5, 5, GomokuPlayerBlack);

	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 0, 2, GomokuPlayerWhite);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 1, 1, GomokuPlayerWhite);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 3, 5, GomokuPlayerWhite);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 4, 2, GomokuPlayerWhite);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 4, 4, GomokuPlayerWhite);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 5, 3, GomokuPlayerWhite);

	md->itsGameIsOver = true;
	
	md->itsGameOf.Gomoku2D.itsGameStatus		= GomokuGameWin;
	md->itsGameOf.Gomoku2D.itsWinLine.itsH		= -0.5 + (1.0/GOMOKU_SIZE)*5;
	md->itsGameOf.Gomoku2D.itsWinLine.itsV		= -0.5 + (1.0/GOMOKU_SIZE)*3;
	md->itsGameOf.Gomoku2D.itsWinLine.itsFlip	= false;	//	ignored
	md->itsGameOf.Gomoku2D.itsWinLine.itsAngle	= 0.75*PI;
	md->itsGameOf.Gomoku2D.itsWinLine.itsSizeH	= ROOT2 * 4.0 / GOMOKU_SIZE;		//	line length, excluding endcaps
	md->itsGameOf.Gomoku2D.itsWinLine.itsSizeV	= HALO_FACTOR * (1.0 / GOMOKU_SIZE);//	line thickness

#endif

#ifdef MAKE_GAME_CHOICE_ICONS

	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 2, 1, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 2, 4, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 3, 4, GomokuPlayerBlack);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 4, 3, GomokuPlayerBlack);

	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 1, 3, GomokuPlayerWhite);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 2, 3, GomokuPlayerWhite);
	AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, 3, 2, GomokuPlayerWhite);

#endif
}

static void InitEmptyBoard(
	GomokuBoard		*aBoard,
	TopologyType	aTopology)
{
	unsigned int		theNumLinesFoundSoFar[GOMOKU_SIZE][GOMOKU_SIZE],
						h,
						v;
	GomokuIntersection	*theIntersection;
	GomokuLine			*theLine;
	unsigned int		i,
						d;	//	d is the five-in-a-row line's direction,
							//		as an index into gGomokuDirections.
	signed int			hh,	//	(hh,vv) moves along the win line.
						vv,
						dh,
						dv;
	unsigned int		b,	//	number of black stones ∈ {0,1,2,3,4,5}
						w;	//	number of white stones ∈ {0,1,2,3,4,5}

	//	Each intersection sits on 20 distinct five-in-a-row lines.
	//	Keep track of how many of them we've found so far.
	for (h = 0; h < GOMOKU_SIZE; h++)
		for (v = 0; v < GOMOKU_SIZE; v++)
			theNumLinesFoundSoFar[h][v] = 0;

	//	The intersections are initially empty.
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			theIntersection = &aBoard->itsIntersections[h][v];
			
			theIntersection->itsStone = GomokuPlayerNone;
			
			//	The code below will locate the 20 five-in-a-row lines
			//	that pass through this intersection.  For now just
			//	set them all to NULL.
			for (i = 0; i < 20; i++)
				theIntersection->itsLines[i] = NULL;
		}
	}
	aBoard->itsTotalNumStones[GomokuPlayerNone ] = GOMOKU_SIZE * GOMOKU_SIZE;
	aBoard->itsTotalNumStones[GomokuPlayerBlack] = 0;
	aBoard->itsTotalNumStones[GomokuPlayerWhite] = 0;
	
	//	Locate all the five-in-a-row lines.
	//	As we find each one, give the intersections it contains
	//	a pointer to it.
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			for (d = 0; d < 4; d++)
			{
				theLine = &aBoard->itsFiveInARowLines[h][v][d];

				//	The lines' five intersections are all empty.
				theLine->itsNumStones[GomokuPlayerNone ] = 5;
				theLine->itsNumStones[GomokuPlayerBlack] = 0;
				theLine->itsNumStones[GomokuPlayerWhite] = 0;
				
				//	Now let's find the GomokuIntersections that sit on this line...
				
				//	Start at the centerpoint, looking "forward".
				hh = h;
				vv = v;
				dh = + gGomokuDirections[d][0];
				dv = + gGomokuDirections[d][1];
				
				//	Record the centerpoint.
				aBoard->itsIntersections[h][v].itsLines[theNumLinesFoundSoFar[h][v]++] = theLine;
				
				//	Move to and record the next two points in the "forward" direction.
				
				AdvanceOneStep(&hh, &vv, &dh, dv, aTopology);
				aBoard->itsIntersections[hh][vv].itsLines[theNumLinesFoundSoFar[hh][vv]++] = theLine;
				
				AdvanceOneStep(&hh, &vv, &dh, dv, aTopology);
				aBoard->itsIntersections[hh][vv].itsLines[theNumLinesFoundSoFar[hh][vv]++] = theLine;
				
				//	Return to the centerpoint, but looking "backward".
				hh = h;
				vv = v;
				dh = - gGomokuDirections[d][0];
				dv = - gGomokuDirections[d][1];
				
				//	Move to and record the next two points in the "backward" direction.

				AdvanceOneStep(&hh, &vv, &dh, dv, aTopology);
				aBoard->itsIntersections[hh][vv].itsLines[theNumLinesFoundSoFar[hh][vv]++] = theLine;
				
				AdvanceOneStep(&hh, &vv, &dh, dv, aTopology);
				aBoard->itsIntersections[hh][vv].itsLines[theNumLinesFoundSoFar[hh][vv]++] = theLine;
			}
		}
	}

	//	Did we find 20 distinct five-in-a-row lines
	//	passing through each intersection?
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			GEOMETRY_GAMES_ASSERT(
				theNumLinesFoundSoFar[h][v] == 20,
				"Failed to find 20 lines at each intersection");
		}
	}
	
	//	All five-in-a-row lines are initially empty.
	for (b = 0; b < 6; b++)		//	number of black stones within a five-in-a-row line
		for (w = 0; w < 6; w++)	//	number of white stones within a five-in-a-row line
			aBoard->itsOccupancyTallies[b][w] = 0;	//	will get overwritten for b = w = 0
	aBoard->itsOccupancyTallies[0][0] = GOMOKU_SIZE * GOMOKU_SIZE * 4;
}

static void AdvanceOneStep(
	signed int		*hh,
	signed int		*vv,
	signed int		*dh,	//	dh might change sign when crossing a Klein bottle seam.
	signed int		 dv,	//	dv never changes sign.
	TopologyType	aTopology)
{
	//	Advance one space.
	*hh += *dh;
	*vv +=  dv;
	
	//	Wrap vertically.
	if (*vv >= GOMOKU_SIZE)
	{
		*vv -= GOMOKU_SIZE;
		if (aTopology == Topology2DKlein)
		{
			*hh = GOMOKU_SIZE - *hh;
			*dh = - *dh;
		}
	}
	if (*vv < 0)
	{
		*vv += GOMOKU_SIZE;
		if (aTopology == Topology2DKlein)
		{
			*hh = GOMOKU_SIZE - *hh;
			*dh = - *dh;
		}
	}
	
	//	Wrap horizontally.
	if (*hh >= GOMOKU_SIZE)
		*hh -= GOMOKU_SIZE;
	if (*hh <  0)
		*hh += GOMOKU_SIZE;
}

static void AddOrRemoveStone(
	GomokuBoard		*aBoard,
	unsigned int	h,
	unsigned int	v,
	GomokuPlayer	aNewStone)	//	none, black or white
{
	GomokuIntersection	*theIntersection;
	GomokuLine			**theLine;
	GomokuPlayer		theOldStone;
	unsigned int		i,
						*theNumStones;
	
	//	Note:  This function will get called from within
	//	our recursion, so we should try to make it
	//	as efficient as possible.
	
	theIntersection	= &aBoard->itsIntersections[h][v];
	theOldStone		= theIntersection->itsStone;

	//	Place the new stone
	theIntersection->itsStone = aNewStone;
	
	//	Adjust the overall stone totals
	aBoard->itsTotalNumStones[theOldStone]--;
	aBoard->itsTotalNumStones[ aNewStone ]++;
	
	//	Adjust the totals on each five-in-a-row line,
	//	and also the corresponding occupancy tallies.
	theLine = theIntersection->itsLines;
	for (i = 20; i-- > 0; )
	{
		theNumStones = (*theLine)->itsNumStones;
		
		aBoard->itsOccupancyTallies
			[theNumStones[GomokuPlayerBlack]]
			[theNumStones[GomokuPlayerWhite]]
			--;

		theNumStones[theOldStone]--;
		theNumStones[ aNewStone ]++;

		aBoard->itsOccupancyTallies
			[theNumStones[GomokuPlayerBlack]]
			[theNumStones[GomokuPlayerWhite]]
			++;

		theLine++;
	}
}


static bool GomokuDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	double			theIntegerPart;	//	value is ignored
	unsigned int	h,
					v;

	UNUSED_PARAMETER(aRightClick);

	//	If the game is already over, ignore all hits.
	if (md->itsGameIsOver)
		return false;

	//	Are we near an intersection?
	//	If not, treat this hit as a scroll.
	if (modf(GOMOKU_SIZE*(md->its2DHandPlacement.itsH + 0.5) + 0.25, &theIntegerPart) > 0.5
	 || modf(GOMOKU_SIZE*(md->its2DHandPlacement.itsV + 0.5) + 0.25, &theIntegerPart) > 0.5)
	{
		//	If the board is still empty, the user might not know
		//	that stones go on intersections, not in cells,
		//	and might be trying in vain to place a stone in a cell.
		if (BoardIsEmpty(md))
		{
			md->itsDragMessage.itsMessage	= GetLocalizedText(u"GomokuStonePlacement");
			md->itsDragMessage.itsTitle		= GetLocalizedText(u"Gomoku");
		}

		return false;
	}

	//	Which intersection got hit?
	FindHitIntersection(&md->its2DHandPlacement, md->itsTopology, &h, &v);

	//	If the intersection is already occupied, treat this hit as a scroll.
	if (md->itsGameOf.Gomoku2D.itsBoard.itsIntersections[h][v].itsStone != GomokuPlayerNone)
		return false;

	//	During a content drag, remember which intersection was hit originally,
	//	and accept the Gomoku move iff the drag ends over that same intersection.
	//	This gives the user a chance to change his/her mind,
	//	and also helps protect against accidentally making a move
	//	when trying to scroll.
	md->itsGameOf.Gomoku2D.itsHitIntersectionH = h;
	md->itsGameOf.Gomoku2D.itsHitIntersectionV = v;

	return true;
}


static void GomokuDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	md->its2DHandPlacement.itsH += (md->its2DHandPlacement.itsFlip ? -aHandLocalDeltaH : aHandLocalDeltaH);
	md->its2DHandPlacement.itsV += aHandLocalDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);
}


static void GomokuDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	unsigned int	h,
					v;

	UNUSED_PARAMETER(aDragDuration);

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized),
	//	return without making a move.
	if (aTouchSequenceWasCancelled)
		return;

	//	Near which intersection did the drag end?
	FindHitIntersection(&md->its2DHandPlacement, md->itsTopology, &h, &v);

	//	Accept the move iff the drag ended near the same intersection where it began.
	if (h == md->itsGameOf.Gomoku2D.itsHitIntersectionH
	 && v == md->itsGameOf.Gomoku2D.itsHitIntersectionV)
	{
		//	Place a black or white stone at the chosen intersection,
		//	according to whose turn it is.
		AddOrRemoveStone(&md->itsGameOf.Gomoku2D.itsBoard, h, v, md->itsGameOf.Gomoku2D.itsWhoseTurn);

		//	Toggle itsGameOf.Gomoku2D.itsWhoseTurn.
		md->itsGameOf.Gomoku2D.itsWhoseTurn =
			(md->itsGameOf.Gomoku2D.itsWhoseTurn == GomokuPlayerBlack) ?
			GomokuPlayerWhite : GomokuPlayerBlack;

		//	Play the move sound.
		EnqueueSoundRequest(u"GomokuMove.mid");

		if (CheckForWin(md, &md->itsGameOf.Gomoku2D.itsWinLine) != GomokuPlayerNone)
		{
			SimulationBegin(md, Simulation2DGomokuWaitToDisplayWin);
		}
		else
		if (BoardIsFull(md))
		{
			md->itsGameIsOver						= true;
			md->itsGameOf.Gomoku2D.itsGameStatus	= GomokuGameDraw;
		}
		else
		{
			if (md->itsHumanVsComputer)
				ComputerMoves(md);
		}
	}
}

static void FindHitIntersection(
	Placement2D		*aHandPlacement,	//	input
	TopologyType	aTopology,			//	input
	unsigned int	*h,					//	output
	unsigned int	*v)					//	output
{
	//	Find the integer coordinates (h,v) of the chosen intersection.
	*h = (unsigned int) floor(GOMOKU_SIZE*(aHandPlacement->itsH + 0.5) + 0.5);
	*v = (unsigned int) floor(GOMOKU_SIZE*(aHandPlacement->itsV + 0.5) + 0.5);
	if (*v > GOMOKU_SIZE - 1)	//	Test for *v > GOMOKU_SIZE - 1 rather than *v == GOMOKU_SIZE
								//	to protect against invalid input.
	{
		*v = 0;
		if (aTopology == Topology2DKlein)
			*h = GOMOKU_SIZE - *h;
	}
	if (*h > GOMOKU_SIZE - 1)	//	Test for *h > GOMOKU_SIZE - 1 rather than *h == GOMOKU_SIZE
								//	to protect against invalid input.
	{
		*h = 0;
	}
}


static GomokuPlayer	CheckForWin(ModelData *md, Placement2D *aFiveInARow)
{
	//	Does some five-in-a-row line contain five black stones?
	if (md->itsGameOf.Gomoku2D.itsBoard.itsOccupancyTallies[5][0] > 0)
	{
		//	Black has won
		
		//	This call to LocateWinLine() should never fail...
		if ( ! LocateWinLine(&md->itsGameOf.Gomoku2D.itsBoard, 5, 0, aFiveInARow) )
		{
			//	...but let's supply a fallback placement anyhow.
			*aFiveInARow = (Placement2D) {0.0, 0.0, false, 0.0, 0.125, 0.125};
		}
		
		return GomokuPlayerBlack;
	}
		
	//	Does some five-in-a-row line contain five white stones?
	if (md->itsGameOf.Gomoku2D.itsBoard.itsOccupancyTallies[0][5] > 0)
	{
		//	White has won
		
		//	This call to LocateWinLine() should never fail...
		if ( ! LocateWinLine(&md->itsGameOf.Gomoku2D.itsBoard, 0, 5, aFiveInARow) )
		{
			//	...but let's supply a fallback placement anyhow.
			*aFiveInARow = (Placement2D) {0.0, 0.0, false, 0.0, 0.125, 0.125};
		}
		
		return GomokuPlayerWhite;
	}
		
	return GomokuPlayerNone;		//	neither player has won yet
}

static bool LocateWinLine(
	GomokuBoard		*aBoard,
	unsigned int	aNumBlackStones,	//	∈ {0, 5}
	unsigned int	aNumWhiteStones,	//	= 5 - aNumBlackStones
	Placement2D		*aFiveInARow)
{
	unsigned int	h,	//	a win line's centerpoint (h,v) and direction d
					v,
					d;

	//	theWinAngles[] and theWinLengths[] use the same indexing
	//	as gGomokuDirections[], namely E, NE, N, NW.
	static const double	theWinAngles[4]   = {
												0.00*PI,
												0.25*PI,
												0.50*PI,
												0.75*PI
											},
						theWinLengths[4]  = {
														4.0 / GOMOKU_SIZE,
												ROOT2 * 4.0 / GOMOKU_SIZE,
														4.0 / GOMOKU_SIZE,
												ROOT2 * 4.0 / GOMOKU_SIZE
											};

	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			for (d = 0; d < 4; d++)
			{
				if (aBoard->itsFiveInARowLines[h][v][d].itsNumStones[GomokuPlayerBlack] == aNumBlackStones
				 && aBoard->itsFiveInARowLines[h][v][d].itsNumStones[GomokuPlayerWhite] == aNumWhiteStones)
				{
					aFiveInARow->itsH		= -0.5 + (1.0/GOMOKU_SIZE)*h;
					aFiveInARow->itsV		= -0.5 + (1.0/GOMOKU_SIZE)*v;
					aFiveInARow->itsFlip	= false;			//	ignored
					aFiveInARow->itsAngle	= theWinAngles[d];
					aFiveInARow->itsSizeH	= theWinLengths[d];	//	line length, excluding endcaps
					aFiveInARow->itsSizeV	= HALO_FACTOR * (1.0 / GOMOKU_SIZE);	//	line thickness
				
					return true;
				}
			}
		}
	}

	return false;
}

static bool BoardIsFull(ModelData *md)
{
	return (md->itsGameOf.Gomoku2D.itsBoard.itsTotalNumStones[GomokuPlayerNone] == 0);
}

static bool BoardIsEmpty(ModelData *md)
{
	return (md->itsGameOf.Gomoku2D.itsBoard.itsTotalNumStones[GomokuPlayerNone] == GOMOKU_SIZE * GOMOKU_SIZE);
}


static void GomokuSimulationUpdate(ModelData *md)
{
	switch (md->itsSimulationStatus)
	{
		case Simulation2DGomokuChooseComputerMove:
		
			//	While the computer chooses its move in a separate thread,
			//	the main thread is keeping the user interface responsive.
			//	This is good but we don't want to hog too many cycles.
			//	So let's let the main thread take a little nap.
			SleepBriefly();

			//	Wait half a second before letting the computer respond,
			//	for the user's psychological benefit.
			//	Then, once the computer has finished thinking, ...
			if ( md->itsSimulationElapsedTime >= 0.5
			 && ! md->itsGameOf.Gomoku2D.itsThreadThinkingFlag)
			{
				//	...let it make the chosen move.
				SimulationEnd(md);
				ComputerMakesMove(md);
			}

			break;

		case Simulation2DGomokuWaitToDisplayWin:
			//	Pause briefly so the user may appreciate
			//	the move before the winning line appears.
			if (md->itsSimulationElapsedTime >= 0.5)
			{
				SimulationEnd(md);
				md->itsGameIsOver						= true;				//	Record the win.
				md->itsGameOf.Gomoku2D.itsGameStatus	= GomokuGameWin;
				EnqueueSoundRequest(u"GomokuWin.mid");								//	Start a victory sound playing.
				SimulationBegin(md, Simulation2DGomokuFlash);				//	Flash the five-in-a-row marker.
			}
			break;

		case Simulation2DGomokuFlash:
			md->itsFlashFlag = (((unsigned int) floor(10.0 * md->itsSimulationElapsedTime)) % 2) ? true : false;
			if (md->itsSimulationElapsedTime >= 1.4)
			{
				SimulationEnd(md);
				md->itsFlashFlag = false;
			}
			break;

		default:
			break;
	}
}


static void ComputerMoves(ModelData *md)
{
	//	Let the computer do its thinking in a separate thread.
	//	After completing the move it will set itsThreadThinkingFlag = false.
	md->itsGameOf.Gomoku2D.itsThreadThinkingFlag	= true;
	md->itsGameOf.Gomoku2D.itsThreadAbortFlag		= false;
	StartNewThread(md, &ComputerSelectsMove);
	
	//	Let the main thread keep the user interface responsive
	//	while the separate thread selects a move.
	//	If the user cancels the game (for example if s/he
	//	resets Gomoku2D or selects a different game),
	//	set itsThreadAbortFlag to true and wait until
	//	the separate thread sets itsThreadThinkingFlag to false
	//	to signal completion (at which point we are guaranteed
	//	that it's done accessing shared data).
	SimulationBegin(md, Simulation2DGomokuChooseComputerMove);
}

static void ComputerSelectsMove(ModelData *md)
{
	bool			*theThinkingFlag,
					*theAbortFlag;
	unsigned int	*theBestMoveH,
					*theBestMoveV;
	TopologyType	theTopology;
	GomokuPlayer	theWhoseTurn;
	unsigned int	theDifficultyLevel;	//	0…3
	GomokuBoard		theBoard;
	unsigned int	h,
					v;
	GomokuPlayer	theStone;
	signed int		theRecursionDepth;

	//	This function will run in its own separate thread,
	//	while the main thread keeps the UI responsive.

	//	Identify the shared variables
	//	that we'll use to communicate with the main thread.
	theThinkingFlag	= &md->itsGameOf.Gomoku2D.itsThreadThinkingFlag;
	theAbortFlag	= &md->itsGameOf.Gomoku2D.itsThreadAbortFlag;
	theBestMoveH	= &md->itsGameOf.Gomoku2D.itsThreadBestMoveH;
	theBestMoveV	= &md->itsGameOf.Gomoku2D.itsThreadBestMoveV;

	//	Make a copy of the gomoku board
	//	along with other read-only variables.

	theTopology			= md->itsTopology;
	theWhoseTurn		= md->itsGameOf.Gomoku2D.itsWhoseTurn;
	theDifficultyLevel	= md->itsDifficultyLevel;

	InitEmptyBoard(&theBoard, theTopology);
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			theStone = md->itsGameOf.Gomoku2D.itsBoard.itsIntersections[h][v].itsStone;
			if (theStone != GomokuPlayerNone)	//	unnecessary to check, but makes our intentions clear
				AddOrRemoveStone(&theBoard, h, v, theStone);
		}
	}

	//	Hereafter we ignore md.
	//	Our only interaction with its contents is
	//	via the shared variables defined previously.
	md = NULL;

	//	Previous calls to BoardIsFull() should detect
	//	when the board is full, so we may assume
	//	that at least one empty intersection remains.
	//	Nevertheless, let's initialize theBestMove just to be safe.
	*theBestMoveH = 0;
	*theBestMoveV = 0;

	//	Assign a recursion depth to each difficulty level.
	//	Depth 0 is the only level that could be considered Easy.
	//	From there, let's let Medium have depth 1
	//	and Hard have depth 2, to give players some
	//	reasonable chance of winning, and then jump
	//	to depth 5 for Extra Hard.
	//
	switch (theDifficultyLevel)
	{
		case 0:  theRecursionDepth = 0; break;	//	easy
		case 1:  theRecursionDepth = 1; break;	//	medium
		case 2:  theRecursionDepth = 2; break;	//	hard
		case 3:  theRecursionDepth = 5; break;	//	extra hard
		default: theRecursionDepth = 0; break;	//	should never occur
	}
	

	//	Use a recursion to select the best move.
	(void) ValueOfSituation(&theBoard,
							theWhoseTurn,
							theRecursionDepth,
							GOMOKU_INITIAL_ALPHA,
							GOMOKU_INITIAL_BETA,
							theAbortFlag,
							theBestMoveH,
							theBestMoveV);

	//	We're done.  This thread is about to terminate.
	//	As our last act, let the main thread know it may proceed.
	*theThinkingFlag = false;
}


//	Alpha-beta pruning
//
//	ValueOfSituation()'s search algorithm uses αβ pruning.
//	To understand αβ pruning, it's useful to think in terms
//	of the search tree's "fully determined nodes" and
//	"partially determined nodes".
//
//		A "fully determined node" is one for which the value
//		of the resulting board is known, assuming optimal play
//		by both aPlayer and the opponent for all remaining moves.
//		A player who deviates from the optimal sequence of moves
//		might do worse than the determined value, but cannot do better.
//
//		A "partially determined node" is one for which the optimal result
//		has been determined for some (or none) of its available moves,
//		but not yet for all of them.
//
//	The insight of αβ pruning is that even a partially determined node
//	has some information available to it:  All of its ancestors (higher up
//	in the search tree) are also partially determined nodes, and typically
//	they have already examined some of their other children.  In other words,
//	those ancestor nodes have already examined some of the other moves available
//	to the player at that point in the game.  To encapsulate that information,
//	we may define
//
//		α to be the best result that the current player (the player whose possible
//			moves we're evaluating at the current node in the search tree)
//			could have obtained by choosing a different move at a higher node
//			in the search tree (that is, by making a different move
//			earlier in the game), and
//
//		β to be the worst result that the current opponent could
//			limit the current player to, by choosing a different move
//			at a higher node in the search tree.
//
//	Those ancestor nodes are all only partially determined, so α gives only
//	a lower bound on what aPlayer can achieve, and β gives only an upper bound
//	on what the opponent can limit aPlayer to.  Nevertheless, it's useful information:
//	if, while evaluating the moves available at the current node, we find
//	a move whose value exceeds β, we'll know that our opponent will never
//	let us reach this node -- the opponent will instead choose a different move
//	at an ancestor node, to limit us to a result of value β or less.
//	So we can stop processing the current node immediately, without wasting
//	any more time on it.  That's the heart of αβ pruning.
//
//	Note that we made no use of α there.  α comes into play as we work our way
//	down the search tree.  A positive result for us is a negative result for our opponent,
//	that is, for our opponent, α' and β' are given by
//
//		α' = -β
//		β' = -α
//
//	In practice, αβ pruning is wonderfully effective.
//	A search that took two minutes on my iPhone 5s without αβ pruning,
//	takes only 1 second with it.  αβ pruning could be even more effective
//	in combination with some sort of method that would let us examine
//	the most promising moves first, but for an app like the Torus Games,
//	I think keeping the code simple is more important than trying to write
//	the strongest possible Gomoku algorithm.
//
//	Note #4
//	The β cutoff is never reached at the root level of the search tree,
//	so we're sure to evaluate all possible first moves.  If there's more than one
//	optimal first move, the following code chooses one of them at random.
//	Thus the human player won't always get the same response to a given sequence
//	of moves, which adds bit of variety to the game.
//
static unsigned int ValueOfSituation(
	GomokuBoard		*aBoard,			//	input
	GomokuPlayer	aPlayer,			//	input
	unsigned int	aRecursionHeight,	//	input
	signed int		anAlpha,			//	input; see detailed explanation above
	signed int		aBeta,				//	input; see detailed explanation above
	bool			*anAbortFlag,		//	input;  may be NULL
	unsigned int	*aBestMoveH,		//	output; non-NULL only at root level
	unsigned int	*aBestMoveV)		//	output; non-NULL only at root level
{
	GomokuPlayer	theOpponent;
	unsigned int	h,
					v;
	signed int		theBestValue				= GOMOKU_LARGE_NEGATIVE_VALUE;
	unsigned int	theBestValueMultiplicity	= 0;	//	how many different optimal moves share theBestValue;
														//		used only at root level of search tree,
														//		ignored at all other nodes
	signed int		theValue;

	//	Has the user cancelled?
	if (anAbortFlag != NULL && *anAbortFlag)
	{
		if (aBestMoveH != NULL)
			*aBestMoveH = 0;
		if (aBestMoveV != NULL)
			*aBestMoveV = 0;
			
		return 0;
	}
	
	//	Note the opponent's color.
	theOpponent = (aPlayer == GomokuPlayerBlack ? GomokuPlayerWhite : GomokuPlayerBlack);
	
	//	Tentatively place a stone on each empty intersection in turn,
	//	and see which one gives the best result.
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			if (aBoard->itsIntersections[h][v].itsStone == GomokuPlayerNone)
			{
				//	Temporarily place aPlayer's stone at (h,v).
				AddOrRemoveStone(aBoard, h, v, aPlayer);

				//	Evaluate the results of this move.
				
				if (aPlayer == GomokuPlayerBlack ?
						aBoard->itsOccupancyTallies[5][0] > 0 :	//	black has won
						aBoard->itsOccupancyTallies[0][5] > 0 )	//	white has won
				{
					//	aPlayer has won.
					//
					//	In general, whenever we can win, we'd rather do so sooner (fewer moves)
					//	than later (more moves), as a courtesy to our opponent.
					//	Conversely, when we're facing a loss, we'd rather lose later (more moves)
					//	than sooner (fewer moves), to give our opponent a chance to go wrong.
					//	Both these goals are achieved if, rather than reporting
					//
					//		GOMOKU_WIN_BASE_VALUE
					//
					//	we instead report
					//
					//		GOMOKU_WIN_BASE_VALUE + aRecursionHeight
					//
					//	GOMOKU_WIN_BASE_VALUE is already larger than any possible
					//	value that ValueOfBoardAtLeafNode() might return, and adding in
					//	aRecursionHeight makes an earlier win (higher recursion height)
					//	more attractive than a later one (lower recursion height).
					//
					//	And, of course, a slightly larger positive value for one player
					//	is a slightly more negative value for the other, so an early win
					//	by our opponent would be seen as less desirable than a later win
					//	by our opponent.
					//
					theValue = GOMOKU_WIN_BASE_VALUE + aRecursionHeight;
				}
				else
				if (aBoard->itsTotalNumStones[GomokuPlayerNone] == 0)	//	board is full
				{
					//	The board is full and neither player has won.
					//	The game is a draw, so set theValue = 0.
					theValue = 0;
				}
				else
				if (aRecursionHeight == 0)	//	reached desired recursion depth
				{
					//	Evaluate the board.  We're about to return
					//	with no deeper recursion from this node.
					theValue = ValueOfBoardAtLeafNode(aBoard, aPlayer);
				}
				else
				{
					//	Let the recursion run for aRecursionHeight more steps.
					theValue = -ValueOfSituation(	aBoard,
													theOpponent,
													aRecursionHeight - 1,
													-aBeta,
													-anAlpha,
													anAbortFlag,
													NULL,
													NULL);
				}

				//	Remove the temporary stone from (h,v).
				AddOrRemoveStone(aBoard, h, v, GomokuPlayerNone);


				//	Is this move better than all the alternatives examined so far?
				if (theValue > theBestValue)
				{
					theBestValue				= theValue;
					theBestValueMultiplicity	= 0;
					
					//	If we're at the root level of the search tree,
					//	we'll also execute the code immediately below,
					//	bracketed by
					//
					//		if (aBestMoveH != NULL && aBestMoveV != NULL)
					//		{
					//			if (theValue == theBestValue)
					//			{
					//				...
					//			}
					//		}
					//
					//	Among other things, that code will increment
					//	theBestValueMultiplicity to 1.
				}

				//	If we're at the root level of the search tree,
				//	the caller will want to know a move that realizes theBestValue.
				if (aBestMoveH != NULL && aBestMoveV != NULL)	//	true only at root level of search tree
				{
					if (theValue == theBestValue)
					{
						theBestValueMultiplicity++;

						//	Choose randomly among the equally good moves.
						if ( RandomUnsignedInteger() % theBestValueMultiplicity == 0 )
						{
							*aBestMoveH = h;
							*aBestMoveV = v;
						}
					}
				}

				//	Have we found a move that's better than our known lower bound α ?
				if (theBestValue > anAlpha)
				{
					anAlpha = theBestValue;

					if (aBestMoveH != NULL && aBestMoveV != NULL)	//	true only at root level of search tree
					{
						//	We want to fully evaluate all maximally good moves
						//	the root level of the search tree, so that we may
						//	choose one of them randomly (see code above).
						//	The hope is that the human player will enjoy
						//	having the game make different responses
						//	to the same set of initial moves.
						//	In order to keep analyzing moves of equal value
						//	at the root level, it suffices to artifically
						//	decrement α by one.
						anAlpha--;
					}
				
					//	If α ≥ β, then there's no point in evaluating
					//	aPlayer's remaining possible moves at this node,
					//	because our opponent can make a different move,
					//	at a higher node in the search tree, to obtain
					//	a result of β.
					if (anAlpha >= aBeta)
						goto ReturnTheBestValueFoundSoFar;
				}
			}
		}
	}
	
	//	We can reach this point for one of two reasons:
	//	either the nested loops have run their full course,
	//	or we reach a point where α ≥ β and immediately broke out
	//	of the nested loops.
ReturnTheBestValueFoundSoFar:

  	//	Note:  The caller should have already checked that the board wasn't full.
  	//	At the root level, the overall gameplay code notices when the board is full,
  	//	and sets itsGameIsOver = true and itsGameStatus = GomokuGameDraw.
  	//	At all other levels, the code in the nested loops above
  	//	tests whether the board is full, and make a deeper recursive call
  	//	only if it isn't.

	return theBestValue;
}

static signed int ValueOfBoardAtLeafNode(
	GomokuBoard		*aBoard,
	GomokuPlayer	aPlayer)
{
	signed int	theValue;
	
	//	I'm not sure what the best algorithm here would be.
	//	Clearly there's a trade-off between making
	//	it fast and making is powerful.
	//	A more sophisticated algorithm might think
	//	in terms of sequences of forced moves,
	//	but for present purposes I'd rather keep things simple.
	//
	//		Note:  Neither player should have 5 in a row
	//		(the caller has tested for that already)
	//		so we don't consider that possibility here.
	//
	//	For simplicity let's evaluate the board from black's perspective,
	//	and then negate the result at the end if necessary.

	theValue = GOMOKU_4_OF_5_VALUE
				* ((signed int)aBoard->itsOccupancyTallies[4][0] - (signed int)aBoard->itsOccupancyTallies[0][4])
			 +
			   GOMOKU_3_OF_5_VALUE
				* ((signed int)aBoard->itsOccupancyTallies[3][0] - (signed int)aBoard->itsOccupancyTallies[0][3])
			 +
			   GOMOKU_2_OF_5_VALUE
				* ((signed int)aBoard->itsOccupancyTallies[2][0] - (signed int)aBoard->itsOccupancyTallies[0][2]);
	
	if (aPlayer == GomokuPlayerBlack)
		return theValue;
	else
		return - theValue;
}


static void ComputerMakesMove(ModelData *md)
{
	//	Make the move.
	AddOrRemoveStone(
		&md->itsGameOf.Gomoku2D.itsBoard,
		md->itsGameOf.Gomoku2D.itsThreadBestMoveH,
		md->itsGameOf.Gomoku2D.itsThreadBestMoveV,
		md->itsGameOf.Gomoku2D.itsWhoseTurn);

	//	Toggle the player.
	md->itsGameOf.Gomoku2D.itsWhoseTurn = (md->itsGameOf.Gomoku2D.itsWhoseTurn == GomokuPlayerBlack) ? GomokuPlayerWhite : GomokuPlayerBlack;

	//	Start the move sound playing.
	EnqueueSoundRequest(u"GomokuMove.mid");
	
	//	Request a redraw.
	md->itsChangeCount++;

	if (CheckForWin(md, &md->itsGameOf.Gomoku2D.itsWinLine) != GomokuPlayerNone)
	{
		//	Record the win and flash the winning line.
		SimulationBegin(md, Simulation2DGomokuWaitToDisplayWin);
	}
	else
	if (BoardIsFull(md))
	{
		md->itsGameIsOver						= true;
		md->itsGameOf.Gomoku2D.itsGameStatus	= GomokuGameDraw;
	}
}


unsigned int GetNum2DGomokuBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_GOMOKU;
}

void Get2DGomokuKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.50;
	someKleinAxisColors[1][1] = 1.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}

unsigned int GetNum2DGomokuSprites(void)
{
	return GOMOKU_SIZE*GOMOKU_SIZE + 1;	//	intersections plus (possibily unused) win line
}

void Get2DGomokuSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int	i,
					h,
					v;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DGomoku,
		"Game2DGomoku must be active");

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DGomokuSprites(),
		"Internal error:  wrong buffer size");

	//	cells
	//		note (h,v) indexing -- not (row,col)
	for (h = 0; h < GOMOKU_SIZE; h++)
	{
		for (v = 0; v < GOMOKU_SIZE; v++)
		{
			i = GOMOKU_SIZE*h + v;
			aPlacementBuffer[i].itsH		= -0.5 + (double)h * (1.0 / (double)GOMOKU_SIZE);
			aPlacementBuffer[i].itsV		= -0.5 + (double)v * (1.0 / (double)GOMOKU_SIZE);
			aPlacementBuffer[i].itsFlip		= false;
			aPlacementBuffer[i].itsAngle	= 0.0;
			aPlacementBuffer[i].itsSizeH	= 1.0 / GOMOKU_SIZE;
			aPlacementBuffer[i].itsSizeV	= 1.0 / GOMOKU_SIZE;
		}
	}

	//	win line
	if (md->itsGameOf.Gomoku2D.itsGameStatus == GomokuGameWin)
		aPlacementBuffer[GOMOKU_SIZE*GOMOKU_SIZE] = md->itsGameOf.Gomoku2D.itsWinLine;
	else
		aPlacementBuffer[GOMOKU_SIZE*GOMOKU_SIZE] = gUnusedPlacement;
}


static void TerminateThinkingThread(ModelData *md)
{
	//	If a thinking thread exists...
	if (md->itsGameOf.Gomoku2D.itsThreadThinkingFlag)
	{
		//	...politely ask it to stop as quickly as possible and then...
		md->itsGameOf.Gomoku2D.itsThreadAbortFlag = true;
		
		//	...wait for it to do so.  The reason we wait here
		//	for the thread to stop is that we want to rigorously guarantee
		//	that the main thread can't start a new thinking thread
		//	while the old thinking thread is still using 
		//	the shared communication variables.
		while (md->itsGameOf.Gomoku2D.itsThreadThinkingFlag)
			SleepBriefly();
	}
}
